/* SCS simple terminal
    Copyright 2005-2012 SCS GmbH & Co. KG, Hanau, Germany
    written by Peter Mack (peter.mack@scs-ptc.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <linux/serial.h>

#include "lock.h"

// for use with a patched kernel module please comment out the following line
#define P4DRAGON	// set the special baudrate

#define BAUDRATE	B38400	// custom divisor only works if baud is set to 38400 !
//#define BAUDRATE	B115200	// this is the proper baudrate for the SCS PTC-IIIusb and a good choice for the PTC-II modem series

char serdev[] = "/dev/ttyUSB0";
int	ser;
int	run;

void sigHandler (int sig)
{
    if (sig == SIGINT || sig == SIGTERM || sig == SIGQUIT || sig == SIGKILL || sig == SIGHUP)
    {
	run = 0;
    }
}

int main (void)
{
	fd_set	readfds, testfds;
	ssize_t rd;
	char buf[256];
	struct termios	new_termios, old_termios;

	struct serial_struct sstruct;

	printf ("SimpleTerm\n(c) 2005-2012 SCS GmbH & Co. KG, Hanau, Germany\npress CTRL-C to end program\n");

	if (lock_device (serdev) < 0)
	{
		// error
		fprintf (stderr, "Could not lock %s\n", serdev);
		return EXIT_FAILURE;
	}

	if ((ser = open (serdev, O_RDWR | O_NOCTTY)) == -1)
	{
		// Error
		fprintf (stderr, "Could not open %s\n", serdev);
		unlock_device (serdev);
		return EXIT_FAILURE;
	}

	/* save current port status */
	tcgetattr(ser, &old_termios);

	new_termios = old_termios;

	new_termios.c_cc[VTIME] = 0;
	new_termios.c_cc[VMIN] = 1;
	new_termios.c_iflag = 0;
	new_termios.c_iflag |= IGNBRK;
	new_termios.c_oflag = 0;
	new_termios.c_lflag = 0;
	new_termios.c_cflag |= (CS8 | CRTSCTS | CREAD | CLOCAL);
	new_termios.c_cflag &= ~(CSTOPB | PARENB | PARODD | HUPCL);

	cfsetispeed(&new_termios, BAUDRATE);
	cfsetospeed(&new_termios, BAUDRATE);

	tcsetattr(ser, TCSANOW, &new_termios);

#ifdef P4DRAGON
// special P4dragon handling (if not done by the kernel module!)

	// get serial_struct
	if (ioctl(ser, TIOCGSERIAL, &sstruct) < 0)
	{
	    printf("Error: could not get comm ioctl\n");
	    close (ser);
	    unlock_device (serdev);
	    exit(0);
	}

	// set custom divisor to get 829440 baud
	sstruct.custom_divisor = 29;
	sstruct.flags |= ASYNC_SPD_CUST;

	// set serial_struct
	if (ioctl(ser, TIOCSSERIAL, &sstruct) < 0)
	{
	    printf("Error: could not set custom comm baud divisor\n");
	    close (ser);
	    unlock_device (serdev);
	    exit(0);
	}
// end P4dragon handling
#endif

	signal (SIGALRM, sigHandler);
	signal (SIGINT, sigHandler);
	signal (SIGTERM, sigHandler);
	signal (SIGQUIT, sigHandler);
	signal (SIGKILL, sigHandler);
	signal (SIGHUP, sigHandler);

	FD_ZERO (&readfds);
	FD_SET (ser, &readfds);
	FD_SET (0, &readfds);

	write (ser, "\r", 1);

	run = 1;
	while (run)
	{
		testfds = readfds;
		if (select (FD_SETSIZE, &testfds, NULL, NULL, NULL) < 0)
		{
			//fprintf (stderr, "\nSelect error!\n");
			//return EXIT_FAILURE;
			goto EXIT;
		}

		if (FD_ISSET (ser, &testfds))
		{
			rd = read (ser, buf, 256);
			write (1, buf, rd);
		}

		if (FD_ISSET (0, &testfds))
		{
			rd = read (0, buf, 256);
			buf[rd - 1] = '\r';
			write (ser, buf, rd);
		}
	}

EXIT:
	close (ser);
	unlock_device (serdev);

	printf ("\n");

	return EXIT_SUCCESS;
}
